; kernel.asm: the LMOS kernel
; $Id: kernel.asm 11 2005-06-18 23:21:06Z sam $
;
; Copyright: (c) 2005 Sam Hocevar <sam@zoy.org>
;   This program is free software; you can redistribute it and/or
;   modify it under the terms of the Do What The Fuck You Want To
;   Public License, Version 2, as published by Sam Hocevar. See
;   http://sam.zoy.org/wtfpl/COPYING for more details.

PIC_SIZE equ 320*200 ; Picture size
PAL_SIZE equ 3*256   ; Palette size
WAV_SIZE equ 64612   ; Sound sample size
CHUNK_SIZE equ 16384 ; Chunk of sound sample to play at once

org 0
bits 16

_start:
    ; Clean up upon arrival
    push cs
    pop ds

    ; Display an optional message
    mov si, msg
    call print_str

    ; Calibrate delay loop
    call calibrate_delay

    ; Disable interrupts to maximise the user's experience. We don't want
    ; a malicious attacker to press Ctrl-Alt-Del.
    cli

    ; Set video mode
    mov ax,13h
    int 10h

mainloop:
    call next_index_in_ds
    ;call clear_screen
    call display_pic
    call play_sound
    ;call delay_500ms ; Don't delay if we play sound!
    jmp mainloop

clear_screen:
    ; Load the VGA screen segment in the extra segment register
    push 0a000h
    pop es

    ; Blank palette
;    mov edx,3c8h
;    mov al,0
;    out dx,al
;    inc dx
;    mov ecx,PAL_SIZE
;blankpal:
;    out dx,al
;    loop blankpal

    ; Clear screen
    mov di, 0
    mov ax, 0
    mov cx, PIC_SIZE/2 ; word count
    rep stosw
    ret

next_index_in_ds:
    ; Update random seed
    mov si, seed
    mov cx, [si]
    mov ax, cx
    shr cx, 5
    and cx, 7 ; Load old value for future reference
try_again:
    mov bx, 017fah
    add bx, ax
    shr bx, 3
    xor bx, ax
    mov ax, 0273fh
    mul bx
    mov [si], ax ; store back

    ; Compute the current image's segment from our seed
    mov bx, ax
    shr bx, 5
    and bx, 7
    ; Start again if it's 7
    cmp bx, 7
    jz try_again
    ; Start again if it's the same as before
    cmp bx, cx
    jz try_again
    ; Put image segment in ds
    add bx, 2
    shl bx, 12 ; Compute our segment, X000h
    push bx
    pop ds

    ret

display_pic:
    ; Load the VGA screen segment in the extra segment register
    push 0a000h
    pop es

    ; Set palette
    mov dx,3c8h
    mov al,0
    out dx,al
    inc dx
    mov cx,PAL_SIZE
    mov si,palette_data
setpal:
    lodsb
    shr al,2    ; vga palette is 6-bit, not 8
    out dx,al
    loop setpal
    ; Display picture
    mov di, 0
    mov cx,PIC_SIZE/2 ; word count
    rep movsw
    ret

delay_500ms:
;    ; Wait 32 VBL
;    mov ecx, 020h
;waitvbl:
;    mov edx,3dah
;s1: in al,dx
;    test al,8h
;    jnz s1
;s2: in al,dx
;    test al,8h
;    jz s2
;    loop waitvbl

    ; This will slow down emulators
    mov cx, 07fffh
wait1:
    push cx
    mov cx, 0fffh
wait2:
    loop wait2
    pop cx
    loop wait1

    ; This will slow down real computers
    mov ah,86h
    mov cx,0
    mov dx,5000 ; sleep half a second
    int 15h

    ret

print_str:
    mov ah, 0Eh
print:
    mov al, [si]
    cmp al, 0
    jz done
    int 10h
    inc si
    jmp print
done:
    ret

play_sound:
    push 1000h
    pop ds ; it's important to do this first
    mov si, delay
    mov ax, [si]
    push ax ; push delay for later use
    mov si, offset
    mov dx, [si] ; current offset in dx
    mov ax, sample_data
    add ax, dx ; current address
    cmp dx, WAV_SIZE - CHUNK_SIZE
    jae end_of_sample
    mov cx, CHUNK_SIZE ; sample size
    add dx, cx
    mov [si], dx ; store next offset
    jmp middle_of_sample
end_of_sample:
    mov cx, WAV_SIZE
    sub cx, dx ; sample size (size of remaining chunk)
    xor bx, bx
    mov [si], bx ; next offset will be zero
middle_of_sample:
    mov si, ax

    ; ds:si points to the sound file buffer
    ; cx holds the length of the sound buffer
    ; dx holds the calculated delay time
    pop dx ; pop delay
    mov bx, dx

    mov al, 34h
    out 43h, al ; timer 0 mode
    mov al, 36h
    out 40h, al ; 22kHz
    mov al, 90h
    out 43h, al ; timer 2 mode
    in al, 61h
    or al, 03h
    out 61h, al ; enable speaker
playloop:
    lodsb
    out 42h, al
    mov dx, bx
    push cx
delayloop:
    mov cx, 00ffh
innerloop:
    loop innerloop
    dec dx
    jnz delayloop
    pop cx
    loop playloop

    in al, 61h
    and al, 0fch
    out 61h, al ; disable speaker
    ret

calibrate_delay:
    push 0001h ; initialise counter
calibrate_again:
    call print_dot
    mov cx, 0
    mov dx, 0
    mov ah, 1
    int 1ah ; set tick counter

    pop cx
    shl cx, 1
    push cx ; increment counter

bigl00p:
    push cx
    mov dx, 0fffh
delayl00p:
    mov cx, 00ffh
innerl00p:
    loop innerl00p
    dec dx
    jnz delayl00p
    pop cx
    loop bigl00p

    mov ah, 0
    int 1ah ; get tick counter

    cmp dx, 0020h
    jb calibrate_again

    ; Big enough! We have the delay in dx, and the loop count in the stack
    pop cx
    mov ax, cx
    shl ax, 2
    ;add ax, cx ; 4 * count
    mov cx, dx
    xor dx, dx
    div cx ; 4 * count / delay

    mov si, delay
    mov [si], ax ; store delay loop in ax
    ret

;print_dx:
;    push cx
;    push ax
;    push dx
;    mov cx, 4
;print_digit:
;    rol dx, 4
;    mov ax, 0e0fh
;    and al, dl
;    add al, 090h
;    daa
;    adc al, 040h
;    daa
;    int 10h
;    loop print_digit
;    pop dx
;    pop ax
;    pop cx
;    ret

print_dot:
    push ax
    mov ax, 0e2eh
    int 10h
    pop ax
    ret

delay:
    dw 0
seed:
    dw 05f7fh ; Seed the random generator so that first image is watermelon.jpg
offset:
    dw 0 ; Offset into the sample

msg:
    db 0, 13, 10, "Calibrating delay loop", 0 ; disabled

sample_data:
    incbin "drums.wav", 0, WAV_SIZE
    times (64*1024 - WAV_SIZE) db 0 ; pad as if it were an image

palette_data:
    %include "watermelon.jpg.raw.pal"
image_data:
    incbin "watermelon.jpg.raw",0,PIC_SIZE

    times (64*1024 - PIC_SIZE - PAL_SIZE) / 4 dw 0,0
    %include "chicken.jpg.raw.pal"
    incbin "chicken.jpg.raw",0,PIC_SIZE

    times (64*1024 - PIC_SIZE - PAL_SIZE) / 4 dw 0,0
    %include "kool-aid.jpg.raw.pal"
    incbin "kool-aid.jpg.raw",0,PIC_SIZE

    times (64*1024 - PIC_SIZE - PAL_SIZE) / 4 dw 0,0
    %include "eazy.jpg.raw.pal"
    incbin "eazy.jpg.raw",0,PIC_SIZE

    times (64*1024 - PIC_SIZE - PAL_SIZE) / 4 dw 0,0
    %include "2pac.jpg.raw.pal"
    incbin "2pac.jpg.raw",0,PIC_SIZE

    times (64*1024 - PIC_SIZE - PAL_SIZE) / 4 dw 0,0
    %include "lilwayne.jpg.raw.pal"
    incbin "lilwayne.jpg.raw",0,PIC_SIZE

    times (64*1024 - PIC_SIZE - PAL_SIZE) / 4 dw 0,0
    %include "purple-drank.jpg.raw.pal"
    incbin "purple-drank.jpg.raw",0,PIC_SIZE

; Stolen from the Goatse floppy
the_end equ ($-_start)
size_in_sectors equ ((the_end - the_end % 512)/512+1)
times (size_in_sectors*512)-($-$$) db 0 ; pad the file so it fills the last disc sector

